// DOM Elements
const importForm = document.getElementById('importForm');
const albumSelect = document.getElementById('albumSelect');
const jsonInput = document.getElementById('jsonInput');
const jsonFile = document.getElementById('jsonFile');
const previewBtn = document.getElementById('previewBtn');
const previewSection = document.getElementById('previewSection');
const previewContent = document.getElementById('previewContent');
const clearBtn = document.getElementById('clearBtn');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    loadAlbums();
    setupEventListeners();
});

// Event Listeners
function setupEventListeners() {
    importForm.addEventListener('submit', handleImport);
    previewBtn.addEventListener('click', handlePreview);
    clearBtn.addEventListener('click', handleClear);
    jsonFile.addEventListener('change', handleFileUpload);
}

// Load albums
async function loadAlbums() {
    try {
        const response = await fetch('scraper_api.php?action=list_albums');
        const albums = await response.json();
        
        albums.forEach(album => {
            const option = document.createElement('option');
            option.value = album.id;
            option.textContent = album.title;
            albumSelect.appendChild(option);
        });
    } catch (error) {
        console.error('Load albums error:', error);
    }
}

// Handle file upload
function handleFileUpload(e) {
    const file = e.target.files[0];
    if (!file) return;
    
    const reader = new FileReader();
    reader.onload = (event) => {
        try {
            const json = JSON.parse(event.target.result);
            jsonInput.value = JSON.stringify(json, null, 2);
        } catch (error) {
            alert('Lỗi: File JSON không hợp lệ!');
            jsonFile.value = '';
        }
    };
    reader.readAsText(file);
}

// Handle preview
function handlePreview() {
    const jsonText = jsonInput.value.trim();
    
    if (!jsonText) {
        alert('Vui lòng nhập JSON data!');
        return;
    }
    
    try {
        const jsonData = JSON.parse(jsonText);
        displayPreview(jsonData);
    } catch (error) {
        alert('Lỗi: JSON không hợp lệ!\n' + error.message);
    }
}

// Display preview
function displayPreview(data) {
    const title = data.title || `${data.cosplayer || ''} cosplay ${data.character || ''}`.trim() || 'Untitled Post';
    const images = data.images || [];
    const videos = data.videos || [];
    
    let previewHtml = `
        <div class="info-grid">
            <div class="info-item">
                <label>Title</label>
                <div class="value">${escapeHtml(title)}</div>
            </div>
            <div class="info-item">
                <label>Cosplayer</label>
                <div class="value">${escapeHtml(data.cosplayer || 'N/A')}</div>
            </div>
            <div class="info-item">
                <label>Character</label>
                <div class="value">${escapeHtml(data.character || 'N/A')}</div>
            </div>
            <div class="info-item">
                <label>Source</label>
                <div class="value">${escapeHtml(data.source || 'N/A')}</div>
            </div>
            <div class="info-item">
                <label>Số ảnh</label>
                <div class="value">${images.length}</div>
            </div>
            <div class="info-item">
                <label>Số video</label>
                <div class="value">${videos.length}</div>
            </div>
            <div class="info-item">
                <label>Tags</label>
                <div class="value">${data.tags && data.tags.length > 0 ? data.tags.join(', ') : 'N/A'}</div>
            </div>
            <div class="info-item">
                <label>Download Links</label>
                <div class="value">${data.download_links && data.download_links.length > 0 ? data.download_links.length + ' links' : 'N/A'}</div>
            </div>
        </div>
    `;
    
    if (images.length > 0) {
        previewHtml += `
            <h3 style="margin-top: 20px;"><i class="fas fa-images"></i> Preview Ảnh (${Math.min(6, images.length)} / ${images.length})</h3>
            <div class="image-preview-grid">
                ${images.slice(0, 6).map(img => `
                    <div class="image-preview-item">
                        <img src="${img}" alt="Preview" onerror="this.src='https://via.placeholder.com/150?text=Error'">
                    </div>
                `).join('')}
            </div>
        `;
    }
    
    previewContent.innerHTML = previewHtml;
    previewSection.style.display = 'block';
    
    // Scroll to preview
    previewSection.scrollIntoView({ behavior: 'smooth' });
}

// Handle import
async function handleImport(e) {
    e.preventDefault();
    
    const albumId = albumSelect.value;
    const jsonText = jsonInput.value.trim();
    
    if (!albumId) {
        alert('Vui lòng chọn album!');
        return;
    }
    
    if (!jsonText) {
        alert('Vui lòng nhập JSON data!');
        return;
    }
    
    let jsonData;
    try {
        jsonData = JSON.parse(jsonText);
    } catch (error) {
        alert('Lỗi: JSON không hợp lệ!\n' + error.message);
        return;
    }
    
    // Prepare post data
    const postData = {
        album_id: albumId,
        title: jsonData.title || `${jsonData.cosplayer || ''} cosplay ${jsonData.character || ''}`.trim() || 'Untitled Post',
        cosplayer: jsonData.cosplayer || '',
        character: jsonData.character || '',
        source: jsonData.source || '',
        images: jsonData.images || [],
        videos: jsonData.videos || [],
        tags: jsonData.tags || [],
        download_links: jsonData.download_links || [],
        photo_count: jsonData.photo_count || jsonData.images?.length || 0,
        video_count: jsonData.video_count || jsonData.videos?.length || 0,
        source_url: jsonData.source_url || '',
        categories: jsonData.categories || []
    };
    
    if (postData.images.length === 0 && postData.videos.length === 0) {
        alert('Post phải có ít nhất 1 ảnh hoặc 1 video!');
        return;
    }
    
    try {
        const submitBtn = importForm.querySelector('button[type="submit"]');
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Đang import...';
        
        const response = await fetch('albums_api.php?action=create_post', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(postData)
        });
        
        const result = await response.json();
        
        if (result.success) {
            alert('Import thành công!');
            window.location.href = `album_detail.php?id=${albumId}`;
        } else {
            alert('Lỗi: ' + (result.error || 'Không thể import post'));
        }
    } catch (error) {
        console.error('Import error:', error);
        alert('Lỗi khi import: ' + error.message);
    } finally {
        const submitBtn = importForm.querySelector('button[type="submit"]');
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fas fa-upload"></i> Import';
    }
}

// Handle clear
function handleClear() {
    jsonInput.value = '';
    jsonFile.value = '';
    previewSection.style.display = 'none';
}

// Utility
function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
