<?php
session_start();

// Kiểm tra quyền admin (có thể thêm authentication sau)
// if (!isset($_SESSION['admin']) || $_SESSION['admin'] !== true) {
//     die('Access denied');
// }

header('Content-Type: application/json');

$action = $_GET['action'] ?? '';

if ($action === 'backup') {
    try {
        $backupDir = __DIR__ . '/backups/';
        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        $timestamp = date('Y-m-d_H-i-s');
        $backupName = 'backup_' . $timestamp;
        $backupPath = $backupDir . $backupName;
        
        // Tạo thư mục backup
        if (!mkdir($backupPath, 0755, true)) {
            throw new Exception('Không thể tạo thư mục backup');
        }
        
        // Backup uploads folder
        $uploadsDir = __DIR__ . '/uploads/';
        if (is_dir($uploadsDir)) {
            $uploadsBackup = $backupPath . '/uploads';
            if (!mkdir($uploadsBackup, 0755, true)) {
                throw new Exception('Không thể tạo thư mục uploads backup');
            }
            
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($uploadsDir, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::SELF_FIRST
            );
            
            foreach ($files as $file) {
                $filePath = $file->getRealPath();
                $relativePath = str_replace($uploadsDir, '', $filePath);
                $destPath = $uploadsBackup . '/' . $relativePath;
                
                if ($file->isDir()) {
                    if (!is_dir($destPath)) {
                        mkdir($destPath, 0755, true);
                    }
                } else {
                    copy($filePath, $destPath);
                }
            }
        }
        
        // Backup config files
        $configFiles = ['r2_config.php', 'config.php'];
        foreach ($configFiles as $configFile) {
            $sourceFile = __DIR__ . '/' . $configFile;
            if (file_exists($sourceFile)) {
                copy($sourceFile, $backupPath . '/' . $configFile);
            }
        }
        
        // Backup PHP files (trừ vendor, backups)
        $phpFiles = glob(__DIR__ . '/*.php');
        $excludeFiles = ['backup.php', 'vendor'];
        foreach ($phpFiles as $phpFile) {
            $fileName = basename($phpFile);
            if (!in_array($fileName, $excludeFiles)) {
                copy($phpFile, $backupPath . '/' . $fileName);
            }
        }
        
        // Tạo file info
        $info = [
            'backup_date' => date('Y-m-d H:i:s'),
            'timestamp' => $timestamp,
            'files_count' => countFiles($backupPath),
            'size' => getDirSize($backupPath)
        ];
        
        file_put_contents($backupPath . '/backup_info.json', json_encode($info, JSON_PRETTY_PRINT));
        
        // Tạo ZIP file
        $zipFile = $backupDir . $backupName . '.zip';
        $zip = new ZipArchive();
        if ($zip->open($zipFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
            addFolderToZip($backupPath, $zip, $backupName);
            $zip->close();
            
            // Xóa thư mục backup sau khi nén
            deleteDirectory($backupPath);
            
            echo json_encode([
                'success' => true,
                'message' => 'Backup thành công!',
                'backup_file' => basename($zipFile),
                'backup_size' => formatBytes(filesize($zipFile)),
                'info' => $info
            ]);
        } else {
            throw new Exception('Không thể tạo file ZIP');
        }
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    
} elseif ($action === 'list') {
    // Liệt kê các backup
    $backupDir = __DIR__ . '/backups/';
    $backups = [];
    
    if (is_dir($backupDir)) {
        $files = glob($backupDir . 'backup_*.zip');
        foreach ($files as $file) {
            $backups[] = [
                'filename' => basename($file),
                'size' => formatBytes(filesize($file)),
                'date' => date('Y-m-d H:i:s', filemtime($file))
            ];
        }
        
        // Sắp xếp theo thời gian (mới nhất trước)
        usort($backups, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });
    }
    
    echo json_encode([
        'success' => true,
        'backups' => $backups,
        'count' => count($backups)
    ]);
    
} elseif ($action === 'download') {
    // Download backup
    $filename = $_GET['file'] ?? '';
    if (empty($filename)) {
        http_response_code(400);
        echo json_encode(['error' => 'Tên file không hợp lệ']);
        exit;
    }
    
    $filePath = __DIR__ . '/backups/' . basename($filename);
    if (!file_exists($filePath) || !preg_match('/^backup_.*\.zip$/', basename($filename))) {
        http_response_code(404);
        echo json_encode(['error' => 'File không tồn tại']);
        exit;
    }
    
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . basename($filePath) . '"');
    header('Content-Length: ' . filesize($filePath));
    readfile($filePath);
    exit;
    
} elseif ($action === 'delete') {
    // Xóa backup
    $filename = $_GET['file'] ?? '';
    if (empty($filename)) {
        http_response_code(400);
        echo json_encode(['error' => 'Tên file không hợp lệ']);
        exit;
    }
    
    $filePath = __DIR__ . '/backups/' . basename($filename);
    if (!file_exists($filePath) || !preg_match('/^backup_.*\.zip$/', basename($filename))) {
        http_response_code(404);
        echo json_encode(['error' => 'File không tồn tại']);
        exit;
    }
    
    if (unlink($filePath)) {
        echo json_encode(['success' => true, 'message' => 'Đã xóa backup thành công']);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Không thể xóa file']);
    }
    
} else {
    http_response_code(400);
    echo json_encode(['error' => 'Action không hợp lệ']);
}

// Helper functions
function countFiles($dir) {
    $count = 0;
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    foreach ($files as $file) {
        if ($file->isFile()) {
            $count++;
        }
    }
    return $count;
}

function getDirSize($dir) {
    $size = 0;
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    foreach ($files as $file) {
        if ($file->isFile()) {
            $size += $file->getSize();
        }
    }
    return $size;
}

function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, $precision) . ' ' . $units[$pow];
}

function addFolderToZip($folder, &$zip, $basePath = '') {
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($folder, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    
    foreach ($files as $file) {
        $filePath = $file->getRealPath();
        $relativePath = str_replace($folder . '/', '', $filePath);
        $zipPath = $basePath . '/' . $relativePath;
        
        if ($file->isDir()) {
            $zip->addEmptyDir($zipPath);
        } else {
            $zip->addFile($filePath, $zipPath);
        }
    }
}

function deleteDirectory($dir) {
    if (!is_dir($dir)) {
        return false;
    }
    
    $files = array_diff(scandir($dir), array('.', '..'));
    foreach ($files as $file) {
        $filePath = $dir . '/' . $file;
        is_dir($filePath) ? deleteDirectory($filePath) : unlink($filePath);
    }
    return rmdir($dir);
}
?>

