<?php
header('Content-Type: application/json');

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'scrape':
        scrapeCosplaytele();
        break;
    case 'list_albums':
        listAlbums();
        break;
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
        break;
}

/**
 * Scrape thông tin từ cosplaytele.com
 */
function scrapeCosplaytele() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    if (!$data) {
        $data = $_POST;
    }
    
    $url = $data['url'] ?? '';
    
    if (empty($url)) {
        http_response_code(400);
        echo json_encode(['error' => 'URL không được để trống']);
        return;
    }
    
    // Validate URL
    if (!filter_var($url, FILTER_VALIDATE_URL) || strpos($url, 'cosplaytele.com') === false) {
        http_response_code(400);
        echo json_encode(['error' => 'URL không hợp lệ hoặc không phải cosplaytele.com']);
        return;
    }
    
    // Fetch HTML content
    $html = fetchUrl($url);
    
    if (!$html) {
        http_response_code(500);
        echo json_encode(['error' => 'Không thể lấy nội dung từ URL']);
        return;
    }
    
    // Parse HTML
    $result = parseCosplaytelePage($html, $url);
    
    if (!$result || isset($result['error'])) {
        http_response_code(500);
        echo json_encode(['error' => $result['error'] ?? 'Lỗi khi parse nội dung']);
        return;
    }
    
    echo json_encode($result);
}

/**
 * Fetch URL content
 */
function fetchUrl($url) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
        CURLOPT_TIMEOUT => 30,
        CURLOPT_CONNECTTIMEOUT => 10
    ]);
    
    $html = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$html) {
        return false;
    }
    
    return $html;
}

/**
 * Parse cosplaytele.com page
 */
function parseCosplaytelePage($html, $url) {
    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    @$dom->loadHTML('<?xml encoding="UTF-8">' . $html);
    $xpath = new DOMXPath($dom);
    
    $result = [
        'source_url' => $url,
        'cosplayer' => '',
        'character' => '',
        'source' => '',
        'photo_count' => 0,
        'video_count' => 0,
        'download_links' => [],
        'tags' => [],
        'categories' => [],
        'images' => []
    ];
    
    // Extract title
    $titleNodes = $xpath->query("//h1 | //h2 | //h3 | //h4 | //h5 | //h6[contains(@class, 'entry-title')]");
    if ($titleNodes->length > 0) {
        $title = trim($titleNodes->item(0)->textContent);
        $result['title'] = $title;
    }
    
    // Extract cosplayer, character, source from content
    $contentText = $dom->textContent;
    
    // Pattern: Cosplayer: [name]
    if (preg_match('/Cosplayer:\s*([^\n\r<]+)/i', $html, $matches)) {
        $result['cosplayer'] = trim(strip_tags($matches[1]));
    }
    
    // Pattern: Character: [name]
    if (preg_match('/Character:\s*([^\n\r<]+)/i', $html, $matches)) {
        $result['character'] = trim(strip_tags($matches[1]));
    }
    
    // Pattern: Appear In: [name] or Source: [name]
    if (preg_match('/(?:Appear In|Source):\s*([^\n\r<]+)/i', $html, $matches)) {
        $result['source'] = trim(strip_tags($matches[1]));
    }
    
    // Extract photo count
    if (preg_match('/(\d+)\s*photos?/i', $html, $matches)) {
        $result['photo_count'] = (int)$matches[1];
    }
    
    // Extract video count
    if (preg_match('/(\d+)\s*videos?/i', $html, $matches)) {
        $result['video_count'] = (int)$matches[1];
    }
    
    // Extract download links (Mediafire, Telegram, etc.)
    $linkNodes = $xpath->query("//a[contains(@href, 'mediafire') or contains(@href, 'telegram') or contains(@href, 'gofile') or contains(@href, 'sorafolder')]");
    foreach ($linkNodes as $node) {
        $href = $node->getAttribute('href');
        $text = trim($node->textContent);
        if ($href && $text) {
            $result['download_links'][] = [
                'type' => getDownloadType($text),
                'url' => $href,
                'text' => $text
            ];
        }
    }
    
    // Extract tags
    $tagNodes = $xpath->query("//a[contains(@rel, 'tag')] | //span[contains(@class, 'tag')]//a");
    foreach ($tagNodes as $node) {
        $tag = trim($node->textContent);
        if ($tag) {
            $result['tags'][] = $tag;
        }
    }
    
    // Extract categories
    $catNodes = $xpath->query("//a[contains(@rel, 'category')] | //span[contains(@class, 'cat')]//a");
    foreach ($catNodes as $node) {
        $cat = trim($node->textContent);
        if ($cat && !in_array($cat, $result['categories'])) {
            $result['categories'][] = $cat;
        }
    }
    
    // Extract images
    $imgNodes = $xpath->query("//img[contains(@src, 'cosplaytele') or contains(@src, 'cdn')] | //img[contains(@class, 'attachment')]");
    $imagesFound = [];
    foreach ($imgNodes as $img) {
        $src = $img->getAttribute('src');
        if ($src) {
            // Convert relative URLs to absolute
            if (strpos($src, 'http') !== 0) {
                $parsedUrl = parse_url($url);
                $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
                if (strpos($src, '/') === 0) {
                    $src = $baseUrl . $src;
                } else {
                    $src = $baseUrl . '/' . $src;
                }
            }
            
            // Filter out small images (icons, avatars)
            if (strpos($src, 'avatar') === false && 
                strpos($src, 'icon') === false &&
                strpos($src, 'logo') === false) {
                $imagesFound[] = $src;
            }
        }
    }
    
    // Also try to extract from data attributes
    $dataImgNodes = $xpath->query("//*[@data-src or @data-lazy-src]");
    foreach ($dataImgNodes as $node) {
        $src = $node->getAttribute('data-src') ?: $node->getAttribute('data-lazy-src');
        if ($src && strpos($src, 'http') === 0) {
            $imagesFound[] = $src;
        }
    }
    
    // Remove duplicates and limit
    $result['images'] = array_values(array_unique(array_slice($imagesFound, 0, 100)));
    
    // Clean empty values
    $result['tags'] = array_values(array_unique(array_filter($result['tags'])));
    $result['categories'] = array_values(array_unique(array_filter($result['categories'])));
    $result['download_links'] = array_values($result['download_links']);
    
    return $result;
}

/**
 * Get download type from text
 */
function getDownloadType($text) {
    $text = strtolower($text);
    if (strpos($text, 'mediafire') !== false) return 'mediafire';
    if (strpos($text, 'telegram') !== false) return 'telegram';
    if (strpos($text, 'gofile') !== false) return 'gofile';
    if (strpos($text, 'sorafolder') !== false) return 'sorafolder';
    return 'other';
}

/**
 * List albums for dropdown
 */
function listAlbums() {
    $albumsDir = __DIR__ . '/albums/';
    
    if (!is_dir($albumsDir)) {
        echo json_encode([]);
        return;
    }
    
    $albums = [];
    $files = scandir($albumsDir);
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..' || pathinfo($file, PATHINFO_EXTENSION) !== 'json') {
            continue;
        }
        
        $filepath = $albumsDir . $file;
        $data = json_decode(file_get_contents($filepath), true);
        
        if ($data && isset($data['id'])) {
            $albums[] = [
                'id' => $data['id'],
                'title' => $data['title'] ?? 'Untitled Album'
            ];
        }
    }
    
    // Sort by title
    usort($albums, function($a, $b) {
        return strcmp($a['title'], $b['title']);
    });
    
    echo json_encode($albums);
}
?>
